/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/type.hpp
 *
 * @brief [LEVEL: beta] definition of the message and record types:
 * @ref diagnostics::Type_t
 *
 * $Id: type.hpp,v 1.15 2005/06/23 09:54:20 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @test diagnostics/frame/type.t.cpp
 */

#ifndef DIAGNOSTICS__FRAME__TYPE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__TYPE_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

/**
 * @brief The message types determine the meaning of a log message. It
 * has no consequences on format of the fields of a @ref
 * ::diagnostics::Record -- but it does determine (together with other
 * parameters) whether is treated as an error-indication by the
 * unittest framework (see @ref
 * ::diagnostics::unittest::Test_Run_Result for details).
 */
typedef enum {
    /**
     * Entering a block. What is free to the user.
     */
    TYPE_BLOCK_ENTER=0,
    /**
     * Leaving a block -- all values (including file_name, base_file_name and line)
     * are identical to those of the corresponding TYPE_BLOCK_ENTER.
     */
    TYPE_BLOCK_EXIT=1,

    /**
     * Entering and leaving a free procedure. What contians the name of
     * the method -- and can be use log the invocation parameters in
     * addition.
     */
    TYPE_PROCEDURE_ENTER=2,
    /**
     * Leaving a procedure -- all values (including file_name, base_file_name and line)
     * are identical to those of the corresponding TYPE_PROCEDURE_ENTER.
     */
    TYPE_PROCEDURE_EXIT=3,

    /**
     * Entering and leaving a class method. What contians the name of
     * the method -- and can be use log the invocation parameters in
     * addition.
     */
    TYPE_METHOD_ENTER=4,
    /**
     * Leaving a method -- all values (including file_name, base_file_name and line)
     * are identical to those of the corresponding TYPE_METHOD_ENTER.
     */
    TYPE_METHOD_EXIT=5,

    /**
     * Human readable trace message.
     */
    TYPE_TRACE=6,

    /**
     * Binary readable trace message.
     */
    TYPE_TRACE_BINARY=7,

    /**
     * A failed assertion, i.e., a error condition which was not
     * expected to happen.
     *
     * This is always treated as an error by the unittest of
     * diagnostics (see unittest.hpp) independently of the @ref
     * Level_t.
     */
    TYPE_FAILED_ASSERTION=8,

    /**
     * A failed check, i.e., a error condition which was 
     * expected to happen due to unproper use of a module or due
     * to external circumstances (e.g. network down). 
     *
     * Whehter such an event is treated as an error or not by the
     * unittest of diagnostics (see unittest.hpp) depends on the level
     * of the event and the current build level (see @ref
     * unittest::Run_Test_Suite_Traversal). However, if such a message
     * at level @ref LEVEL_TEST occurs, a test is treated as invalid,
     * i.e., it was not possible to execute the test to come to a
     * conclusive result.
    */
    TYPE_FAILED_CHECK=9,

    /**
     * An exception occured when none was expected. What described the
     * unexpected exception.
     *
     * Unittest treats this an error, independently of the level.
     */
    TYPE_UNEXPECTED_EXCEPTION=10,

    /**
     * An exception occured when another one was expected. The what 
     *
     * Unittest treats this an error iff it occurs at @ref LEVEL_TEST.
     */
    TYPE_WRONG_EXCEPTION=11,

    /**
     * No exception occured when an exception was expected. The what
     * string describes the expected exception.
     *
     * Unittest treats this an error iff it occurs at @ref LEVEL_TEST.
     */
    TYPE_MISSING_EXCEPTION=12,

    /**
     * A testcase started. What contains the testcase name.
     */
    TYPE_TESTCASE_ENTER=13,
    /**
     * Leaving a testcase -- all values (including file_name,
     * base_file_name and line) are identical to those of the
     * corresponding TYPE_TESTCASE_ENTER.
     */
    TYPE_TESTCASE_EXIT=14,

    /**
     * Whenever a logger is registered, it first receives a message of
     * this type at @ref LEVEL_SYSTEM. All fields are blank with the
     * exception of name, what and line.
     *
     * If the logger is one of the initial ones (set by @ref
     * set_initial_loggers), then name says so, what contains a time
     * stamp of the start up, and line is equal to 1.
     *
     * If the logger is not an initial one, then the name of message
     * says so, what contains two time stamps (framework activation
     * and registration) and line is to 0.
     */
    TYPE_LOG_OPEN=15,
    
    /**
     * Whenever a logger is unregistered, it receives beforehand such
     * a message at @ref LEVEL_SYSTEM. 
	 *
	 * The what field inidicates whether the whole framework is
	 * shutting down or whether the logger is unregistered.
     */
    TYPE_LOG_CLOSE=16
} Type_t;

/**
 * @brief The maximum integer value of @ref diagnostics::Type_t
 */
#define TYPE_MAX TYPE_LOG_CLOSE


/**
 * @brief Returns meaningful string for a record type. If the
 * type is unkown, i.e., a value not defined in the enum is passed,
 * "UNKOWN" is returned.
 *
 * @throw never throws (not declared for efficiency)
 */
inline char const * type_to_string(Type_t const type) 
{
    char const * const type_map[]= 
	{ "block_enter",
	  "block_exit",
	  
	  "procedure_enter",
	  "procedure_exit",
				
	  "method_enter",
	  "method_exit",
				
	  "trace",
	  "trace_binary",
				
	  "failed_assertion",
	  "failed_check",
				
	  "unexpected_exception",
	  "wrong_exception",
	  "missing_exception",

	  "testcase_enter",
	  "testcase_exit",

	  "log_open",
	  "log_close",
 
          "UNKNOWN"};

    return (type>TYPE_MAX ? type_map[TYPE_MAX+1] : type_map[type]);
}


DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
